import { z } from "zod"

// Validação para login
export const loginSchema = z.object({
  email: z
    .string()
    .min(1, "Email é obrigatório")
    .email("Email deve ter um formato válido")
    .max(255, "Email deve ter no máximo 255 caracteres"),
  password: z
    .string()
    .min(1, "Senha é obrigatória")
    .min(6, "Senha deve ter pelo menos 6 caracteres")
    .max(100, "Senha deve ter no máximo 100 caracteres"),
})

// Validação para criação de usuário
export const createUserSchema = z.object({
  name: z
    .string()
    .min(1, "Nome é obrigatório")
    .min(2, "Nome deve ter pelo menos 2 caracteres")
    .max(255, "Nome deve ter no máximo 255 caracteres")
    .regex(/^[a-zA-ZÀ-ÿ\s]+$/, "Nome deve conter apenas letras e espaços"),
  email: z
    .string()
    .min(1, "Email é obrigatório")
    .email("Email deve ter um formato válido")
    .max(255, "Email deve ter no máximo 255 caracteres"),
  password: z
    .string()
    .min(1, "Senha é obrigatória")
    .min(6, "Senha deve ter pelo menos 6 caracteres")
    .max(100, "Senha deve ter no máximo 100 caracteres"),
})

// Validação para criação de grupo de repertório
export const createRepertoireGroupSchema = z.object({
  name: z
    .string()
    .min(1, "Nome é obrigatório")
    .min(2, "Nome deve ter pelo menos 2 caracteres")
    .max(255, "Nome deve ter no máximo 255 caracteres"),
  description: z
    .string()
    .max(1000, "Descrição deve ter no máximo 1000 caracteres")
    .optional(),
  color: z
    .string()
    .regex(/^#[0-9A-F]{6}$/i, "Cor deve estar no formato hexadecimal (#RRGGBB)")
    .optional(),
  userId: z
    .number()
    .int("ID do usuário deve ser um número inteiro")
    .positive("ID do usuário deve ser positivo"),
})

// Validação para criação de música
export const createSongSchema = z.object({
  title: z
    .string()
    .min(1, "Título é obrigatório")
    .min(2, "Título deve ter pelo menos 2 caracteres")
    .max(255, "Título deve ter no máximo 255 caracteres"),
  artist: z
    .string()
    .min(1, "Artista é obrigatório")
    .min(2, "Artista deve ter pelo menos 2 caracteres")
    .max(255, "Artista deve ter no máximo 255 caracteres"),
  youtubeUrl: z
    .string()
    .optional()
    .refine((val) => {
      if (!val || val.trim() === "") return true
      try {
        const youtubeRegex = /^(https?:\/\/)?(www\.)?(youtube\.com\/watch\?v=|youtu\.be\/)[\w-]+(\?[\w=&-]*)?$/
        return youtubeRegex.test(val.trim())
      } catch (error) {
        console.error('Erro na validação da URL do YouTube:', error)
        return false
      }
    }, "URL deve ser válida e do YouTube"),
  repertoireGroupId: z
    .number()
    .int("ID do grupo deve ser um número inteiro")
    .positive("ID do grupo deve ser positivo"),
  // categoryId removido temporariamente até a tabela categories ser criada
})

// Validação para atualização de música
export const updateSongSchema = createSongSchema.partial().extend({
  id: z
    .number()
    .int("ID deve ser um número inteiro")
    .positive("ID deve ser positivo"),
})

// Validação para parâmetros de query
export const userIdQuerySchema = z.object({
  userId: z
    .string()
    .min(1, "ID do usuário é obrigatório")
    .transform((val) => parseInt(val, 10))
    .refine((val) => !isNaN(val) && val > 0, "ID do usuário deve ser um número positivo"),
})

// Tipos TypeScript derivados dos schemas
export type LoginInput = z.infer<typeof loginSchema>
export type CreateUserInput = z.infer<typeof createUserSchema>
export type CreateRepertoireGroupInput = z.infer<typeof createRepertoireGroupSchema>
export type CreateSongInput = z.infer<typeof createSongSchema>
export type UpdateSongInput = z.infer<typeof updateSongSchema>
export type UserIdQuery = z.infer<typeof userIdQuerySchema>

// Função helper para validação
export function validateData<T>(schema: z.ZodSchema<T>, data: unknown): { success: true; data: T } | { success: false; errors: string[] } {
  try {
    const validatedData = schema.parse(data)
    return { success: true, data: validatedData }
  } catch (error) {
    if (error instanceof z.ZodError) {
      const errors = error.errors.map((err) => `${err.path.join('.')}: ${err.message}`)
      return { success: false, errors }
    }
    return { success: false, errors: ["Erro de validação desconhecido"] }
  }
}

// Função específica para validação de query params
export function validateQueryData<T>(schema: z.ZodSchema<T>, data: any): { success: true; data: T } | { success: false; errors: string[] } {
  try {
    const validatedData = schema.parse(data)
    return { success: true, data: validatedData }
  } catch (error) {
    if (error instanceof z.ZodError) {
      const errors = error.errors.map((err) => `${err.path.join('.')}: ${err.message}`)
      return { success: false, errors }
    }
    return { success: false, errors: ["Erro de validação desconhecido"] }
  }
}