"use client"

interface AuthData {
  id: number
  name: string
  email: string
  type: "master" | "user"
  expires: number
}

const AUTH_KEY = "repertorio_auth"

export class AuthClient {
  private static instance: AuthClient
  private authData: AuthData | null = null

  private constructor() {
    if (typeof window !== "undefined") {
      this.loadFromStorage()
    }
  }

  static getInstance(): AuthClient {
    if (!AuthClient.instance) {
      AuthClient.instance = new AuthClient()
    }
    return AuthClient.instance
  }

  private loadFromStorage(): void {
    try {
      const stored = sessionStorage.getItem(AUTH_KEY)
      if (stored) {
        const data = JSON.parse(stored) as AuthData
        if (data.expires > Date.now()) {
          this.authData = data
        } else {
          this.clearAuth()
        }
      }
    } catch (error) {
      console.error("[v0] Error loading auth from storage:", error)
      this.clearAuth()
    }
  }

  setAuth(data: AuthData): void {
    this.authData = data
    if (typeof window !== "undefined") {
      try {
        sessionStorage.setItem(AUTH_KEY, JSON.stringify(data))
      } catch (error) {
        console.error("[v0] Error saving auth to storage:", error)
      }
    }
  }

  getAuth(): AuthData | null {
    if (this.authData && this.authData.expires > Date.now()) {
      return this.authData
    }
    this.clearAuth()
    return null
  }

  clearAuth(): void {
    this.authData = null
    if (typeof window !== "undefined") {
      try {
        sessionStorage.removeItem(AUTH_KEY)
      } catch (error) {
        console.error("[v0] Error clearing auth storage:", error)
      }
    }
  }

  isAuthenticated(): boolean {
    return this.getAuth() !== null
  }

  getUserType(): "master" | "user" | null {
    const auth = this.getAuth()
    return auth ? auth.type : null
  }
}

export const authClient = AuthClient.getInstance()

export function getClientUser() {
  const auth = authClient.getAuth()
  return auth
}

export function clearClientAuth() {
  authClient.clearAuth()
}
