'use server'

import { neon } from "@neondatabase/serverless"
import { createSongSchema, validateData } from "@/lib/validations"
import { revalidatePath } from "next/cache"

const sql = neon(process.env.DATABASE_URL!)

export async function addSongAction(formData: {
  title: string
  artist: string
  youtubeUrl: string
  repertoireGroupId: number
}) {
  try {
    // Validar dados de entrada
    const validation = validateData(createSongSchema, formData)
    if (!validation.success) {
      return {
        success: false,
        error: "Dados inválidos",
        details: validation.errors
      }
    }

    const { title, artist, youtubeUrl, repertoireGroupId } = validation.data

    // Extract YouTube video ID
    let youtubeVideoId = ""
    if (youtubeUrl) {
      const match = youtubeUrl.match(/(?:youtube\.com\/watch\?v=|youtu\.be\/)([^&\n?#]+)/)
      youtubeVideoId = match ? match[1] : ""
    }

    // Get next position
    const positionResult = await sql`
      SELECT COALESCE(MAX(position), 0) + 1 as next_position
      FROM songs
      WHERE repertoire_group_id = ${repertoireGroupId}
    `

    const nextPosition = positionResult[0].next_position

    const result = await sql`
      INSERT INTO songs (title, artist, youtube_url, youtube_video_id, duration, position, repertoire_group_id)
      VALUES (${title}, ${artist}, ${youtubeUrl || ""}, ${youtubeVideoId}, ${0}, ${nextPosition}, ${repertoireGroupId})
      RETURNING id, title, artist, youtube_url, youtube_video_id, duration, position, repertoire_group_id, created_at
    `

    // Revalidate the page to show new data
    revalidatePath('/user-dashboard')

    return {
      success: true,
      song: result[0],
    }
  } catch (error) {
    console.error("Create song error:", error)
    return {
      success: false,
      error: "Erro interno do servidor",
    }
  }
}