'use client';

import { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Label } from '@/components/ui/label';
import { Music, Search, Plus, Filter, ShoppingCart } from 'lucide-react';

interface Song {
  id: number;
  title: string;
  artist: string;
  key: string;
  bpm: number;
  duration: string;
  lyrics: string;
  category_id: number;
  category_name?: string;
  category_color?: string;
  created_by_name: string;
  created_at: string;
}

interface Category {
  id: number;
  name: string;
  description: string;
  color: string;
}

interface RepertoireGroup {
  id: number;
  name: string;
}

interface MusicStoreModalProps {
  isOpen: boolean;
  onClose: () => void;
  onAddToRepertoire?: (songId: number, groupId: number, position: number) => Promise<void>;
}

export default function MusicStoreModal({ isOpen, onClose, onAddToRepertoire }: MusicStoreModalProps) {
  const [songs, setSongs] = useState<Song[]>([]);
  const [categories, setCategories] = useState<Category[]>([]);
  const [repertoireGroups, setRepertoireGroups] = useState<RepertoireGroup[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<string>('all');
  
  // Estados para adicionar música ao repertório
  const [selectedSong, setSelectedSong] = useState<Song | null>(null);
  const [selectedGroup, setSelectedGroup] = useState<string>('');
  const [position, setPosition] = useState<string>('');
  const [addingToRepertoire, setAddingToRepertoire] = useState(false);
  const [showAddDialog, setShowAddDialog] = useState(false);
  const [showLyricsDialog, setShowLyricsDialog] = useState(false);

  useEffect(() => {
    if (isOpen) {
      loadSongs();
      loadCategories();
      loadRepertoireGroups();
    }
  }, [isOpen]);

  const loadSongs = async () => {
    try {
      const response = await fetch('/api/music-store');
      const data = await response.json();
      if (data.success) {
        setSongs(data.songs);
      }
    } catch (error) {
      console.error('Erro ao carregar músicas:', error);
    } finally {
      setLoading(false);
    }
  };

  const loadCategories = async () => {
    try {
      const response = await fetch('/api/categories');
      const data = await response.json();
      if (data.success) {
        setCategories(data.categories);
      }
    } catch (error) {
      console.error('Erro ao carregar categorias:', error);
    }
  };

  const loadRepertoireGroups = async () => {
    try {
      const response = await fetch('/api/repertoires');
      const data = await response.json();
      if (data.success) {
        setRepertoireGroups(data.repertoires);
      }
    } catch (error) {
      console.error('Erro ao carregar grupos de repertório:', error);
    }
  };

  const handleAddToRepertoire = async () => {
    if (!selectedSong || !selectedGroup || !position) {
      alert('Por favor, preencha todos os campos');
      return;
    }

    setAddingToRepertoire(true);
    try {
      if (onAddToRepertoire) {
        await onAddToRepertoire(selectedSong.id, parseInt(selectedGroup), parseInt(position));
      } else {
        // Fallback para adicionar diretamente via API
        const response = await fetch('/api/songs', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            title: selectedSong.title,
            artist: selectedSong.artist,
            key: selectedSong.key,
            bpm: selectedSong.bpm,
            duration: selectedSong.duration,
            lyrics: selectedSong.lyrics,
            category_id: selectedSong.category_id,
            repertoire_group_id: parseInt(selectedGroup),
            position: parseInt(position),
          }),
        });

        const data = await response.json();
        if (!data.success) {
          throw new Error(data.error);
        }
      }
      
      alert('Música adicionada ao repertório com sucesso!');
      setSelectedSong(null);
      setSelectedGroup('');
      setPosition('');
      setShowAddDialog(false);
    } catch (error) {
      console.error('Erro ao adicionar música ao repertório:', error);
      alert('Erro ao adicionar música ao repertório');
    } finally {
      setAddingToRepertoire(false);
    }
  };

  const filteredSongs = songs.filter(song => {
    const matchesSearch = song.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         song.artist.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesCategory = selectedCategory === 'all' || song.category_id?.toString() === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  if (loading) {
    return (
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="max-w-6xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Carregando</DialogTitle>
            <DialogDescription>Aguarde enquanto carregamos a loja de músicas...</DialogDescription>
          </DialogHeader>
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <Music className="h-12 w-12 text-blue-600 mx-auto mb-4 animate-pulse" />
              <p className="text-gray-600">Carregando loja de músicas...</p>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    );
  }

  return (
    <>
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="max-w-6xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <div className="flex items-center space-x-3">
              <ShoppingCart className="h-6 w-6 text-blue-600" />
              <DialogTitle className="text-2xl">Loja de Músicas</DialogTitle>
              <Badge variant="outline" className="text-sm">
                {filteredSongs.length} músicas disponíveis
              </Badge>
            </div>
            <DialogDescription>
              Explore nossa coleção de músicas e adicione suas favoritas ao seu repertório.
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-6">
            {/* Filtros */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex flex-col md:flex-row gap-4">
                <div className="flex-1">
                  <div className="relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                    <Input
                      placeholder="Buscar por título ou artista..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10"
                    />
                  </div>
                </div>
                <div className="md:w-64">
                  <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                    <SelectTrigger>
                      <Filter className="h-4 w-4 mr-2" />
                      <SelectValue placeholder="Filtrar por categoria" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">Todas as categorias</SelectItem>
                      {categories.map((category) => (
                        <SelectItem key={category.id} value={category.id.toString()}>
                          <div className="flex items-center space-x-2">
                            <div 
                              className="w-3 h-3 rounded-full" 
                              style={{ backgroundColor: category.color }}
                            />
                            <span>{category.name}</span>
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </div>

            {/* Grid de Músicas */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 max-h-96 overflow-y-auto">
              {filteredSongs.map((song) => (
                <Card key={song.id} className="p-4 hover:shadow-lg transition-shadow">
                  <div className="flex items-start justify-between mb-3">
                    <div className="flex-1">
                      <h3 className="font-semibold text-base text-gray-900 mb-1">
                        {song.title}
                      </h3>
                      <p className="text-gray-600 text-sm mb-2">{song.artist}</p>
                      {song.category_name && (
                        <Badge 
                          variant="outline" 
                          className="text-xs"
                          style={{ 
                            borderColor: song.category_color,
                            color: song.category_color 
                          }}
                        >
                          {song.category_name}
                        </Badge>
                      )}
                    </div>
                    <Music className="h-5 w-5 text-gray-400" />
                  </div>

                  <div className="space-y-1 mb-3 text-xs text-gray-600">
                    <div className="flex justify-between">
                      <span>Tom:</span>
                      <span className="font-medium">{song.key}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>BPM:</span>
                      <span className="font-medium">{song.bpm}</span>
                    </div>
                    <div className="flex justify-between">
                      <span>Duração:</span>
                      <span className="font-medium">{song.duration}</span>
                    </div>
                  </div>

                  <div className="flex space-x-2">
                    <Button 
                      variant="outline" 
                      size="sm" 
                      className="flex-1 text-xs"
                      onClick={() => {
                        setSelectedSong(song);
                        setShowLyricsDialog(true);
                      }}
                    >
                      Ver Letra
                    </Button>
                    <Button 
                      size="sm" 
                      className="flex-1 text-xs"
                      onClick={() => {
                        setSelectedSong(song);
                        setShowAddDialog(true);
                      }}
                    >
                      <Plus className="h-3 w-3 mr-1" />
                      Adicionar
                    </Button>
                  </div>
                </Card>
              ))}
            </div>

            {filteredSongs.length === 0 && (
              <div className="text-center py-8">
                <Music className="h-12 w-12 text-gray-300 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">
                  Nenhuma música encontrada
                </h3>
                <p className="text-gray-600">
                  Tente ajustar os filtros ou termos de busca.
                </p>
              </div>
            )}
          </div>
        </DialogContent>
      </Dialog>

      {/* Modal de Ver Letra */}
      <Dialog open={showLyricsDialog} onOpenChange={setShowLyricsDialog}>
        <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>
              {selectedSong?.title} - {selectedSong?.artist}
            </DialogTitle>
            <DialogDescription>
              Letra completa da música selecionada.
            </DialogDescription>
          </DialogHeader>
          <div className="mt-4">
            <pre className="whitespace-pre-wrap text-sm leading-relaxed">
              {selectedSong?.lyrics}
            </pre>
          </div>
        </DialogContent>
      </Dialog>

      {/* Modal de Adicionar ao Repertório */}
      <Dialog open={showAddDialog} onOpenChange={setShowAddDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Adicionar ao Repertório</DialogTitle>
            <DialogDescription>
              Selecione o grupo de repertório onde deseja adicionar esta música.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 mt-4">
            <div>
              <Label>Música Selecionada</Label>
              <p className="text-sm text-gray-600 mt-1">
                {selectedSong?.title} - {selectedSong?.artist}
              </p>
            </div>
            
            <div>
              <Label htmlFor="group">Grupo de Repertório</Label>
              <Select value={selectedGroup} onValueChange={setSelectedGroup}>
                <SelectTrigger>
                  <SelectValue placeholder="Selecione um grupo" />
                </SelectTrigger>
                <SelectContent>
                  {repertoireGroups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label htmlFor="position">Posição no Repertório</Label>
              <Input
                id="position"
                type="number"
                placeholder="Ex: 1, 2, 3..."
                value={position}
                onChange={(e) => setPosition(e.target.value)}
                min="1"
              />
            </div>

            <Button 
              onClick={handleAddToRepertoire}
              disabled={addingToRepertoire}
              className="w-full"
            >
              {addingToRepertoire ? 'Adicionando...' : 'Adicionar ao Repertório'}
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </>
  );
}