"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { LogOut, Music, Plus, Play, Edit, Trash2, ShoppingCart } from "lucide-react"
import { useRouter } from "next/navigation"
import MusicPlayer from "@/components/music-player"
import MusicStoreModal from "@/components/music-store-modal"
import { authClient } from "@/lib/auth-client"

interface Song {
  id: number
  title: string
  artist: string
  youtubeUrl: string
  category_id?: number
}

interface Category {
  id: number
  name: string
  description: string
  color: string
}

interface RepertoireGroup {
  id: number
  name: string
  songs: Song[]
  user_id: string
}

export default function UserDashboard() {
  const [user, setUser] = useState<any>(null)
  const [groups, setGroups] = useState<RepertoireGroup[]>([])
  const [categories, setCategories] = useState<Category[]>([])
  const [isGroupDialogOpen, setIsGroupDialogOpen] = useState(false)
  const [isSongDialogOpen, setIsSongDialogOpen] = useState(false)
  const [selectedGroupId, setSelectedGroupId] = useState<number | null>(null)
  const [newGroup, setNewGroup] = useState({ name: "" })
  const [newSong, setNewSong] = useState({ title: "", artist: "", youtubeUrl: "", categoryId: undefined as string | undefined })
  const [draggedSong, setDraggedSong] = useState<{ song: Song; fromGroupId: number } | null>(null)
  const [isLoading, setIsLoading] = useState(false)
  const [isAuthenticated, setIsAuthenticated] = useState(false)

  const [isPlayerOpen, setIsPlayerOpen] = useState(false)
  const [currentPlaylist, setCurrentPlaylist] = useState<Song[]>([])
  const [currentSongIndex, setCurrentSongIndex] = useState(0)

  const [isEditSongDialogOpen, setIsEditSongDialogOpen] = useState(false)
  const [editingSong, setEditingSong] = useState<Song | null>(null)
  const [editSongData, setEditSongData] = useState({ title: "", artist: "", youtubeUrl: "", categoryId: undefined as string | undefined })
  const [isMusicStoreOpen, setIsMusicStoreOpen] = useState(false)

  const router = useRouter()

  useEffect(() => {
    checkAuth()
    loadCategories()
  }, [])

  const checkAuth = async () => {
    console.log("[v0] Dashboard checking auth...")

    try {
      const localAuth = authClient.getAuth()
      if (localAuth) {
        console.log("[v0] Found local auth data, verifying with server")

        const response = await fetch("/api/auth/check", {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({ authData: localAuth }),
        })

        const data = await response.json()

        if (data.authenticated && data.user.type === "user") {
          console.log("[v0] Hybrid auth successful")
          setIsAuthenticated(true)
          setUser(data.user)
          loadGroups(data.user.id)
          return
        }
      }

      const response = await fetch("/api/auth/check")
      const data = await response.json()

      console.log("[v0] Auth check response:", data)

      if (!data.authenticated || data.user.type !== "user") {
        console.log("[v0] Not authenticated or not user, redirecting to login")
        authClient.clearAuth() // Limpa dados locais inválidos
        router.push("/")
        return
      }

      console.log("[v0] Auth successful, setting state")
      setIsAuthenticated(true)
      setUser(data.user)
      loadGroups(data.user.id)
    } catch (error) {
      console.error("Auth check error:", error)
      console.log("[v0] Auth check failed, redirecting to login")
      authClient.clearAuth()
      router.push("/")
    }
  }

  const loadGroups = async (userId: number) => {
    try {
      const response = await fetch(`/api/repertoires?userId=${userId}`)
      const data = await response.json()

      if (data.success && Array.isArray(data.groups)) {
        // Normalizar os dados para garantir que youtube_url seja mapeado para youtubeUrl
        const normalizedGroups = data.groups.map((group: any) => ({
          ...group,
          songs: group.songs.map((song: any) => ({
            ...song,
            youtubeUrl: song.youtube_url || song.youtubeUrl || ""
          }))
        }))
        setGroups(normalizedGroups)
      } else {
        console.log("No groups found or invalid response, setting empty array")
        setGroups([])
      }
    } catch (error) {
      console.error("Error loading groups:", error)
      setGroups([]) // Garantir que groups seja sempre um array
    }
  }

  const loadCategories = async () => {
    try {
      const response = await fetch('/api/categories')
      const data = await response.json()
      if (data.success) {
        setCategories(data.categories)
      }
    } catch (error) {
      console.error('Erro ao carregar categorias:', error)
    }
  }

  const handleLogout = async () => {
    try {
      authClient.clearAuth()
      await fetch("/api/auth/logout", { method: "POST" })
      window.location.href = "/"
    } catch (error) {
      console.error("Logout error:", error)
      authClient.clearAuth()
      window.location.href = "/"
    }
  }

  const handleEditSong = (groupId: number, song: Song) => {
    setEditingSong(song)
    setEditSongData({
      title: song.title,
      artist: song.artist,
      youtubeUrl: song.youtubeUrl,
      categoryId: song.category_id ? song.category_id.toString() : undefined,
    })
    setSelectedGroupId(groupId)
    setIsEditSongDialogOpen(true)
  }

  const handleUpdateSong = async () => {
    if (!editingSong || !selectedGroupId) return

    setIsLoading(true)
    try {
      const response = await fetch(`/api/songs/${editingSong.id}`, {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          title: editSongData.title,
          artist: editSongData.artist,
          youtube_url: editSongData.youtubeUrl,
          categoryId: editSongData.categoryId ? parseInt(editSongData.categoryId) : null,
        }),
      })

      if (response.ok) {
        const updatedSong = await response.json()
        setGroups((prev) =>
          prev.map((group) =>
            group.id === selectedGroupId
              ? {
                  ...group,
                  songs: group.songs.map((song) =>
                    song.id === editingSong.id ? { ...updatedSong, youtubeUrl: updatedSong.youtube_url } : song,
                  ),
                }
              : group,
          ),
        )
        setIsEditSongDialogOpen(false)
        setEditingSong(null)
        setEditSongData({ title: "", artist: "", youtubeUrl: "", categoryId: "" })
      }
    } catch (error) {
      console.error("Erro ao atualizar música:", error)
    } finally {
      setIsLoading(false)
    }
  }

  const handleAddGroup = async (e: React.FormEvent) => {
    e.preventDefault()
    setIsLoading(true)

    try {
      const response = await fetch("/api/repertoires", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          name: newGroup.name,
          userId: user.id,
        }),
      })

      const data = await response.json()

      if (data.success) {
        setGroups([...groups, data.group])
        setNewGroup({ name: "" })
        setIsGroupDialogOpen(false)
      } else {
        alert(data.message)
      }
    } catch (error) {
      console.error("Error creating group:", error)
      alert("Erro ao criar grupo")
    }

    setIsLoading(false)
  }

  const handleAddSong = async () => {
    if (!newSong.title || !newSong.artist || !newSong.youtubeUrl) {
      alert("Por favor, preencha todos os campos obrigatórios")
      return
    }

    try {
      // Use XMLHttpRequest instead of fetch to avoid CORS issues
      const xhr = new XMLHttpRequest()
      
      return new Promise((resolve, reject) => {
        xhr.onreadystatechange = function() {
          if (xhr.readyState === 4) {
            if (xhr.status >= 200 && xhr.status < 300) {
              try {
                const result = JSON.parse(xhr.responseText)
                
                // Reset form and close dialog
                setNewSong({ title: "", artist: "", youtubeUrl: "", categoryId: "" })
                setIsSongDialogOpen(false)
                
                // Refresh to show new song
                window.location.reload()
                resolve(result)
              } catch (parseError) {
                reject(parseError)
              }
            } else {
              reject(new Error(`HTTP ${xhr.status}: ${xhr.statusText}`))
            }
          }
        }
        
        xhr.onerror = function() {
          reject(new Error("Erro de rede"))
        }
        
        xhr.open('POST', '/api/songs', true)
        xhr.setRequestHeader('Content-Type', 'application/json')
        
        const requestData = {
          title: newSong.title,
          artist: newSong.artist,
          youtubeUrl: newSong.youtubeUrl || "",
          repertoireGroupId: selectedGroupId || 1,
          categoryId: newSong.categoryId ? parseInt(newSong.categoryId) : null,
        }
        
        xhr.send(JSON.stringify(requestData))
      })

    } catch (error) {
      console.error("Erro ao adicionar música:", error)
      alert(`Erro ao adicionar música: ${error instanceof Error ? error.message : String(error)}`)
    }
  }

  const handleDeleteGroup = async (groupId: number) => {
    try {
      const response = await fetch(`/api/repertoires/${groupId}`, {
        method: "DELETE",
      })

      const data = await response.json()

      if (data.success) {
        setGroups(groups.filter((group) => group.id !== groupId))
      } else {
        alert(data.message)
      }
    } catch (error) {
      console.error("Error deleting group:", error)
      alert("Erro ao deletar grupo")
    }
  }

  const handleDeleteSong = async (groupId: number, songId: number) => {
    try {
      const response = await fetch(`/api/songs/${songId}`, {
        method: "DELETE",
      })

      const data = await response.json()

      if (data.success) {
        const updatedGroups = groups.map((group) =>
          group.id === groupId ? { ...group, songs: group.songs.filter((song) => song.id !== songId) } : group,
        )
        setGroups(updatedGroups)
      } else {
        alert(data.message)
      }
    } catch (error) {
      console.error("Error deleting song:", error)
      alert("Erro ao deletar música")
    }
  }

  const handleDragStart = (song: Song, fromGroupId: number) => {
    setDraggedSong({ song, fromGroupId })
  }

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault()
  }

  const handleDrop = async (e: React.DragEvent, toGroupId: number) => {
    e.preventDefault()

    if (!draggedSong || draggedSong.fromGroupId === toGroupId) {
      setDraggedSong(null)
      return
    }

    try {
      const response = await fetch(`/api/songs/${draggedSong.song.id}`, {
        method: "PATCH",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          repertoireGroupId: toGroupId,
        }),
      })

      const data = await response.json()

      if (data.success) {
        const updatedGroups = groups.map((group) => {
          if (group.id === draggedSong.fromGroupId) {
            return { ...group, songs: group.songs.filter((song) => song.id !== draggedSong.song.id) }
          }
          if (group.id === toGroupId) {
            return { ...group, songs: [...group.songs, draggedSong.song] }
          }
          return group
        })
        setGroups(updatedGroups)
      } else {
        alert(data.message)
      }
    } catch (error) {
      console.error("Error moving song:", error)
      alert("Erro ao mover música")
    }

    setDraggedSong(null)
  }

  const openAddSongDialog = (groupId: number) => {
    setSelectedGroupId(groupId)
    setIsSongDialogOpen(true)
  }

  const playGroup = (groupId: number) => {
    const group = groups.find((g) => g.id === groupId)
    if (group && group.songs.length > 0) {
      setCurrentPlaylist(group.songs)
      setCurrentSongIndex(0)
      setIsPlayerOpen(true)
    }
  }

  const playSong = (groupId: number, songId: number) => {
    const group = groups.find((g) => g.id === groupId)
    if (group) {
      const songIndex = group.songs.findIndex((s) => s.id === songId)
      if (songIndex !== -1) {
        setCurrentPlaylist(group.songs)
        setCurrentSongIndex(songIndex)
        setIsPlayerOpen(true)
      }
    }
  }

  if (!isAuthenticated || !user) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <Music className="h-12 w-12 text-purple-600 mx-auto mb-4" />
          <p className="text-gray-600">Verificando autenticação...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white shadow-sm border-b">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <Music className="h-8 w-8 text-purple-600 mr-3" />
              <div>
                <h1 className="text-xl font-semibold text-gray-900">Meus Repertórios</h1>
                <p className="text-sm text-gray-500">Bem-vindo, {user.name}</p>
              </div>
            </div>
            <div className="flex items-center space-x-3">
              <Button 
                onClick={() => router.push('/music-store')} 
                variant="outline" 
                size="sm"
              >
                <ShoppingCart className="h-4 w-4 mr-2" />
                Loja de Músicas
              </Button>
              <Button onClick={handleLogout} variant="outline" size="sm">
                <LogOut className="h-4 w-4 mr-2" />
                Sair
              </Button>
            </div>
          </div>
        </div>
      </header>

      {/* Main Content */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <div className="mb-6">
          <div className="flex justify-between items-center">
            <div>
              <h2 className="text-2xl font-bold text-gray-900">Grupos de Repertório</h2>
              <p className="text-gray-600">Organize suas músicas em grupos para seus shows</p>
            </div>

            <Dialog open={isGroupDialogOpen} onOpenChange={setIsGroupDialogOpen}>
              <DialogTrigger asChild>
                <Button>
                  <Plus className="h-4 w-4 mr-2" />
                  Adicionar Grupo
                </Button>
              </DialogTrigger>
              <DialogContent>
                <DialogHeader>
                  <DialogTitle>Criar Novo Grupo</DialogTitle>
                  <DialogDescription>Crie um novo grupo de repertório para organizar suas músicas</DialogDescription>
                </DialogHeader>
                <form onSubmit={handleAddGroup} className="space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="group-name">Nome do Grupo</Label>
                    <Input
                      id="group-name"
                      value={newGroup.name}
                      onChange={(e) => setNewGroup({ name: e.target.value })}
                      placeholder="Ex: Show Acústico, Rock Nacional..."
                      required
                    />
                  </div>
                  <Button type="submit" className="w-full" disabled={isLoading}>
                    {isLoading ? "Criando..." : "Criar Grupo"}
                  </Button>
                </form>
              </DialogContent>
            </Dialog>
          </div>
        </div>

        {/* Kanban Board */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {groups.length === 0 ? (
            <Card className="border-dashed border-2 border-gray-300">
              <CardContent className="flex items-center justify-center py-12">
                <div className="text-center">
                  <Music className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">Nenhum grupo criado</h3>
                  <p className="text-gray-500 mb-4">Crie seu primeiro grupo de repertório</p>
                  <Button onClick={() => setIsGroupDialogOpen(true)}>
                    <Plus className="h-4 w-4 mr-2" />
                    Criar Grupo
                  </Button>
                </div>
              </CardContent>
            </Card>
          ) : (
            groups.map((group) => (
              <Card
                key={group.id}
                className="h-fit"
                onDragOver={handleDragOver}
                onDrop={(e) => handleDrop(e, group.id)}
              >
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-lg">{group.name}</CardTitle>
                    <div className="flex items-center space-x-2">
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => playGroup(group.id)}
                        disabled={group.songs.length === 0}
                      >
                        <Play className="h-4 w-4" />
                      </Button>
                      <Button size="sm" variant="outline" onClick={() => openAddSongDialog(group.id)}>
                        <Plus className="h-4 w-4" />
                      </Button>
                      <Button size="sm" variant="outline" onClick={() => handleDeleteGroup(group.id)}>
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                  <CardDescription>
                    {group.songs.length} música{group.songs.length !== 1 ? "s" : ""}
                  </CardDescription>
                </CardHeader>
                <CardContent className="space-y-2">
                  {group.songs.length === 0 ? (
                    <div className="text-center py-8 text-gray-500">
                      <Music className="h-8 w-8 mx-auto mb-2 opacity-50" />
                      <p className="text-sm">Nenhuma música adicionada</p>
                    </div>
                  ) : (
                    group.songs.map((song) => (
                      <Card
                        key={song.id}
                        className="p-3 cursor-move hover:shadow-md transition-shadow"
                        draggable
                        onDragStart={() => handleDragStart(song, group.id)}
                      >
                        <div className="flex items-center justify-between">
                          <div className="flex-1 min-w-0">
                            <h4 className="font-medium text-sm truncate">{song.title}</h4>
                            <p className="text-xs text-gray-500 truncate">{song.artist}</p>
                          </div>
                          <div className="flex items-center space-x-1 ml-2">
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-8 w-8 p-0"
                              onClick={() => playSong(group.id, song.id)}
                            >
                              <Play className="h-3 w-3" />
                            </Button>
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-8 w-8 p-0"
                              onClick={() => handleEditSong(group.id, song)}
                            >
                              <Edit className="h-3 w-3" />
                            </Button>
                            <Button
                              size="sm"
                              variant="ghost"
                              className="h-8 w-8 p-0"
                              onClick={() => handleDeleteSong(group.id, song.id)}
                            >
                              <Trash2 className="h-3 w-3" />
                            </Button>
                          </div>
                        </div>
                      </Card>
                    ))
                  )}
                </CardContent>
              </Card>
            ))
          )}
        </div>

        {/* Add Song Dialog */}
        <Dialog open={isSongDialogOpen} onOpenChange={setIsSongDialogOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Adicionar Música</DialogTitle>
              <DialogDescription>Adicione uma nova música ao grupo selecionado</DialogDescription>
            </DialogHeader>
            <form onSubmit={handleAddSong} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="song-title">Título da Música</Label>
                <Input
                  id="song-title"
                  value={newSong.title}
                  onChange={(e) => setNewSong({ ...newSong, title: e.target.value })}
                  placeholder="Nome da música"
                  required
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="song-artist">Cantor/Banda</Label>
                <Input
                  id="song-artist"
                  value={newSong.artist}
                  onChange={(e) => setNewSong({ ...newSong, artist: e.target.value })}
                  placeholder="Nome do artista"
                  required
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="song-category">Categoria</Label>
                <Select value={newSong.categoryId || "no-category"} onValueChange={(value) => setNewSong({ ...newSong, categoryId: value === "no-category" ? undefined : value })}>
                  <SelectTrigger>
                    <SelectValue placeholder="Selecione uma categoria (opcional)" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="no-category">Sem categoria</SelectItem>
                    {categories.map((category) => (
                      <SelectItem key={category.id} value={category.id.toString()}>
                        <div className="flex items-center space-x-2">
                          <div 
                            className="w-3 h-3 rounded-full" 
                            style={{ backgroundColor: category.color }}
                          />
                          <span>{category.name}</span>
                        </div>
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="song-url">URL do YouTube</Label>
                <Input
                  id="song-url"
                  type="url"
                  value={newSong.youtubeUrl}
                  onChange={(e) => setNewSong({ ...newSong, youtubeUrl: e.target.value })}
                  placeholder="https://www.youtube.com/watch?v=..."
                  required
                />
              </div>
              <Button type="submit" className="w-full" disabled={isLoading}>
                {isLoading ? "Adicionando..." : "Adicionar Música"}
              </Button>
            </form>
          </DialogContent>
        </Dialog>

        <Dialog open={isEditSongDialogOpen} onOpenChange={setIsEditSongDialogOpen}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Editar Música</DialogTitle>
              <DialogDescription>Edite as informações da música selecionada.</DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <Label htmlFor="edit-title">Título</Label>
                <Input
                  id="edit-title"
                  value={editSongData.title}
                  onChange={(e) => setEditSongData((prev) => ({ ...prev, title: e.target.value }))}
                  placeholder="Nome da música"
                />
              </div>
              <div>
                <Label htmlFor="edit-artist">Artista</Label>
                <Input
                  id="edit-artist"
                  value={editSongData.artist}
                  onChange={(e) => setEditSongData((prev) => ({ ...prev, artist: e.target.value }))}
                  placeholder="Nome do artista"
                />
              </div>
              <div>
                <Label htmlFor="edit-category">Categoria</Label>
                <Select value={editSongData.categoryId || "no-category"} onValueChange={(value) => setEditSongData((prev) => ({ ...prev, categoryId: value === "no-category" ? undefined : value }))}>
                  <SelectTrigger>
                    <SelectValue placeholder="Selecione uma categoria (opcional)" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="no-category">Sem categoria</SelectItem>
                    {categories.map((category) => (
                      <SelectItem key={category.id} value={category.id.toString()}>
                        <div className="flex items-center space-x-2">
                          <div 
                            className="w-3 h-3 rounded-full" 
                            style={{ backgroundColor: category.color }}
                          />
                          <span>{category.name}</span>
                        </div>
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label htmlFor="edit-youtube">URL do YouTube</Label>
                <Input
                  id="edit-youtube"
                  value={editSongData.youtubeUrl}
                  onChange={(e) => setEditSongData((prev) => ({ ...prev, youtubeUrl: e.target.value }))}
                  placeholder="https://www.youtube.com/watch?v=..."
                />
              </div>
              <div className="flex justify-end space-x-2">
                <Button variant="outline" onClick={() => setIsEditSongDialogOpen(false)}>
                  Cancelar
                </Button>
                <Button onClick={handleUpdateSong} disabled={isLoading}>
                  {isLoading ? "Salvando..." : "Salvar"}
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>

        <MusicPlayer
          isOpen={isPlayerOpen}
          onClose={() => setIsPlayerOpen(false)}
          songs={currentPlaylist}
          currentSongIndex={currentSongIndex}
          onSongChange={setCurrentSongIndex}
        />

        <MusicStoreModal
          isOpen={isMusicStoreOpen}
          onClose={() => setIsMusicStoreOpen(false)}
          onAddToRepertoire={async (songId: number, groupId: number, position: number) => {
            try {
              // Primeiro, buscar os dados da música pelo ID
              const songResponse = await fetch(`/api/music-store?id=${songId}`)
              const songData = await songResponse.json()
              
              if (!songData.success || !songData.song) {
                throw new Error('Música não encontrada')
              }

              const song = songData.song

              // Adicionar a música ao repertório
              const response = await fetch('/api/songs', {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                  repertoire_group_id: groupId,
                  position: position,
                  title: song.title,
                  artist: song.artist,
                  key: song.key,
                  bpm: song.bpm,
                  duration: song.duration,
                  lyrics: song.lyrics,
                  category_id: song.category_id,
                }),
              });

              const data = await response.json();
              if (data.success) {
                loadGroups(); // Recarregar os grupos para mostrar a nova música
                setIsMusicStoreOpen(false); // Fechar o modal
              } else {
                throw new Error(data.error);
              }
            } catch (error) {
              console.error('Erro ao adicionar música:', error);
              throw error;
            }
          }}
        />
      </main>
    </div>
  )
}
