import { NextRequest, NextResponse } from "next/server"
import { neon } from "@neondatabase/serverless"
import bcrypt from "bcryptjs"
import { createUserSchema, validateData } from "@/lib/validations"

const sql = neon(process.env.DATABASE_URL!)

export async function GET() {
  try {
    const users = await sql`
      SELECT id, name, email, is_active, is_master, created_at
      FROM users
      ORDER BY created_at DESC
    `

    return NextResponse.json({
      success: true,
      users,
    })
  } catch (error) {
    console.error("Get users error:", error)
    return NextResponse.json(
      {
        success: false,
        message: "Erro interno do servidor",
      },
      { status: 500 },
    )
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    
    // Validar dados de entrada
    const validation = validateData(createUserSchema, body)
    if (!validation.success) {
      return NextResponse.json(
        { 
          error: "Dados inválidos", 
          details: validation.errors 
        },
        { status: 400 }
      )
    }

    const { name, email, password } = validation.data

    // Check if user already exists
    const existingUsers = await sql`
      SELECT id FROM users WHERE email = ${email}
    `

    if (existingUsers.length > 0) {
      return NextResponse.json(
        {
          success: false,
          message: "Usuário já cadastrado com este email",
        },
        { status: 400 },
      )
    }

    // Hash the password before storing
    const saltRounds = 12
    const hashedPassword = await bcrypt.hash(password, saltRounds)

    // Insert new user
    const result = await sql`
      INSERT INTO users (name, email, password_hash, is_active, is_master)
      VALUES (${name}, ${email}, ${hashedPassword}, true, false)
      RETURNING id, name, email, is_active, is_master, created_at
    `

    return NextResponse.json({
      success: true,
      user: result[0],
    })
  } catch (error) {
    console.error("Create user error:", error)
    return NextResponse.json(
      {
        success: false,
        message: "Erro interno do servidor",
      },
      { status: 500 },
    )
  }
}
