import { type NextRequest, NextResponse } from "next/server"
import { neon } from "@neondatabase/serverless"
import { createSongSchema, validateData } from "@/lib/validations"

const sql = neon(process.env.DATABASE_URL!)

export async function POST(request: NextRequest) {
  try {
    // Add CORS headers
    const headers = {
      'Access-Control-Allow-Origin': '*',
      'Access-Control-Allow-Methods': 'POST, OPTIONS',
      'Access-Control-Allow-Headers': 'Content-Type, Accept, Cache-Control',
    }

    const body = await request.json()
    
    // Validar dados de entrada
    const validation = validateData(createSongSchema, body)
    if (!validation.success) {
      return NextResponse.json(
        { 
          success: false,
          error: "Dados inválidos", 
          details: validation.errors 
        },
        { status: 400, headers }
      )
    }

    const { title, artist, youtubeUrl, repertoireGroupId } = validation.data

    // Extract YouTube video ID
    let youtubeVideoId = ""
    if (youtubeUrl) {
      const match = youtubeUrl.match(/(?:youtube\.com\/watch\?v=|youtu\.be\/)([^&\n?#]+)/)
      youtubeVideoId = match ? match[1] : ""
    }

    // Get next position
    const positionResult = await sql`
      SELECT COALESCE(MAX(position), 0) + 1 as next_position
      FROM songs
      WHERE repertoire_group_id = ${repertoireGroupId}
    `

    const nextPosition = positionResult[0].next_position

    const result = await sql`
      INSERT INTO songs (title, artist, youtube_url, youtube_video_id, duration, position, repertoire_group_id, category_id)
      VALUES (${title}, ${artist}, ${youtubeUrl || ""}, ${youtubeVideoId}, ${0}, ${nextPosition}, ${repertoireGroupId}, NULL)
      RETURNING id, title, artist, youtube_url, youtube_video_id, duration, position, repertoire_group_id, category_id, created_at
    `

    return NextResponse.json({
      success: true,
      song: result[0],
    }, { headers })
  } catch (error) {
    console.error("Create song error:", error)
    return NextResponse.json(
      {
        success: false,
        message: "Erro interno do servidor",
      },
      { status: 500, headers: {
        'Access-Control-Allow-Origin': '*',
        'Access-Control-Allow-Methods': 'POST, OPTIONS',
        'Access-Control-Allow-Headers': 'Content-Type, Accept, Cache-Control',
      } },
    )
  }
}

// Handle OPTIONS request for CORS preflight
export async function OPTIONS(request: NextRequest) {
  return new NextResponse(null, {
    status: 200,
    headers: {
      'Access-Control-Allow-Origin': '*',
      'Access-Control-Allow-Methods': 'POST, OPTIONS',
      'Access-Control-Allow-Headers': 'Content-Type, Accept, Cache-Control',
    },
  })
}
