import { NextRequest, NextResponse } from "next/server"
import { neon } from "@neondatabase/serverless"

const sql = neon(process.env.DATABASE_URL!)

// GET - Buscar músicas da loja ou uma música específica por ID
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const id = searchParams.get('id')

    if (id) {
      // Buscar uma música específica por ID
      const result = await sql`
        SELECT 
          s.id,
          s.title,
          s.artist,
          s.youtube_url,
          s.youtube_video_id,
          s.duration,
          s.category_id,
          s.created_at,
          c.name as category_name,
          CASE 
            WHEN s.repertoire_group_id IS NULL THEN 'Loja'
            ELSE rg.name
          END as source
        FROM songs s
        LEFT JOIN music_categories c ON s.category_id = c.id
        LEFT JOIN repertoire_groups rg ON s.repertoire_group_id = rg.id
        WHERE s.id = ${id}
      `
      
      if (result.length === 0) {
        return NextResponse.json({ error: "Música não encontrada" }, { status: 404 })
      }
      
      return NextResponse.json(result[0])
    } else {
      // Buscar todas as músicas cadastradas na aplicação
      const result = await sql`
        SELECT 
          s.id,
          s.title,
          s.artist,
          s.youtube_url,
          s.youtube_video_id,
          s.duration,
          s.category_id,
          s.created_at,
          c.name as category_name,
          CASE 
            WHEN s.repertoire_group_id IS NULL THEN 'Loja'
            ELSE rg.name
          END as source
        FROM songs s
        LEFT JOIN music_categories c ON s.category_id = c.id
        LEFT JOIN repertoire_groups rg ON s.repertoire_group_id = rg.id
        ORDER BY s.created_at DESC
      `
      
      return NextResponse.json(result)
    }
  } catch (error) {
    console.error("Erro ao buscar músicas:", error)
    return NextResponse.json({ error: "Erro interno do servidor" }, { status: 500 })
  }
}

// POST - Adicionar nova música à loja
export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { title, artist, youtube_url, category_id } = body

    // Validação básica
    if (!title || !artist) {
      return NextResponse.json(
        { error: "Título e artista são obrigatórios" },
        { status: 400 }
      )
    }

    // Extrair ID do vídeo do YouTube se URL fornecida
    let youtube_video_id = null
    if (youtube_url) {
      const match = youtube_url.match(/(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/|youtube\.com\/v\/)([^#&?]*)/)
      youtube_video_id = match ? match[1] : null
    }

    // Inserir nova música (sem repertoire_group_id para músicas da loja)
    const result = await sql`
      INSERT INTO songs (title, artist, youtube_url, youtube_video_id, category_id, position)
      VALUES (${title}, ${artist}, ${youtube_url || null}, ${youtube_video_id}, ${category_id || null}, 0)
      RETURNING id, title, artist, youtube_url, youtube_video_id, category_id, created_at
    `

    return NextResponse.json(result[0], { status: 201 })
  } catch (error) {
    console.error("Erro ao adicionar música:", error)
    return NextResponse.json({ error: "Erro interno do servidor" }, { status: 500 })
  }
}