import { NextRequest, NextResponse } from "next/server"
import { neon } from "@neondatabase/serverless"
import bcrypt from "bcryptjs"
import { loginSchema, validateData } from "@/lib/validations"

const sql = neon(process.env.DATABASE_URL!)

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    
    // Validar dados de entrada
    const validation = validateData(loginSchema, body)
    if (!validation.success) {
      return NextResponse.json(
        { 
          error: "Dados inválidos", 
          details: validation.errors 
        },
        { status: 400 }
      )
    }

    const { email, password } = validation.data
    console.log("[v0] Login attempt for email:", email)

    const users = await sql`
      SELECT id, name, email, password_hash, is_active, is_master 
      FROM users 
      WHERE email = ${email} AND is_active = true
    `

    console.log("[v0] Users found:", users.length)

    if (users.length > 0) {
      const user = users[0]
      console.log("[v0] User found:", { id: user.id, email: user.email, is_master: user.is_master })

      if (!(await bcrypt.compare(password, user.password_hash))) {
        console.log("[v0] Password mismatch")
        return NextResponse.json(
          {
            success: false,
            message: "Credenciais inválidas",
          },
          { status: 401 },
        )
      }

      console.log("[v0] Password matches, creating hybrid auth")

      const authData = {
        id: user.id,
        name: user.name,
        email: user.email,
        type: user.is_master ? "master" : "user",
        expires: Date.now() + 24 * 60 * 60 * 1000, // 24 hours
      }

      const response = NextResponse.json({
        success: true,
        user: {
          id: user.id,
          name: user.name,
          email: user.email,
          type: user.is_master ? "master" : "user",
        },
        authData, // Incluindo dados de auth para o cliente
      })

      // Configuração de cookies baseada no ambiente
      const isProduction = process.env.NODE_ENV === 'production'
      
      // Ainda tentamos definir o cookie como fallback
      response.cookies.set("auth_data", JSON.stringify(authData), {
        httpOnly: true, // Sempre httpOnly para segurança
        secure: isProduction, // Secure apenas em produção
        sameSite: "lax",
        path: "/",
        maxAge: 24 * 60 * 60,
      })

      console.log("[v0] Hybrid auth created for user:", user.id)
      console.log("[v0] Auth data:", authData)

      return response
    }

    console.log("[v0] No user found or user inactive")
    return NextResponse.json(
      {
        success: false,
        message: "Credenciais inválidas ou usuário inativo",
      },
      { status: 401 },
    )
  } catch (error) {
    console.error("Login error:", error)
    return NextResponse.json(
      {
        success: false,
        message: "Erro interno do servidor",
      },
      { status: 500 },
    )
  }
}
